﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.EventSystems;
using UnityEngine.UI;

[System.Serializable]
public class SpriteItemsSelector : Selectable, ISelectHandler, IDeselectHandler, IPointerEnterHandler, IPointerExitHandler
{

    //Event in editor to allow code to run when selected
    [SerializeField]
    private Button.ButtonClickedEvent _OnClick = new Button.ButtonClickedEvent();
    
    public Button.ButtonClickedEvent onClick
    {
        get
        {
            return _OnClick;
        }

        set
        {
            _OnClick = value;
        }
    }

    public Sprite[] Sprites;    //The sprites to display
    public int DefaultIndex = 0;    //Allows changing the default displayed sprite

    //UI controls
    public Image Image;
    public Button LeftArrow;
    public Button RightArrow;

    //Animations
    private Animator _Animator;
    private Animator _LeftArrowAnimator;
    private Animator _RightArrowAnimator;

    public int CurrentIndex { get; private set; }   //The index of the currently visible sprite
    public Sprite SelectedSprite { get; private set; }  //The currently visible sprite

    protected override void Awake()
    {
        base.Awake();
        CurrentIndex = DefaultIndex;

        //Get components, subscribe to events
        LeftArrow.onClick.AddListener(OnLeftArrowClick);
        RightArrow.onClick.AddListener(OnRightArrowClick);
        _Animator = GetComponent<Animator>();
        _LeftArrowAnimator = LeftArrow.GetComponent<Animator>();
        _RightArrowAnimator = RightArrow.GetComponent<Animator>();

        //Sanity checks, then display the desired sprite
        if (CurrentIndex >= 0 && CurrentIndex < Sprites.Length)
        {
            SelectedSprite = Sprites[CurrentIndex];
            Image.sprite = SelectedSprite;
        }

        InputManager.Instance.InputStateChanged += OnInputStateChanged;
    }
    
    void Update()
    {
        if (IsInteractable() && currentSelectionState == SelectionState.Selected)
        {
            float val = InputManager.Instance.GetControllerAxisUnique("Controller D-Pad X");

            if (val != 0.0f)
            {
                if (val < -0.2f)
                {
                    //Moved the D-Pad left
                    OnLeftArrowClick();
                }

                else if (val > 0.2f)
                {
                    //Moved the D-Pad right
                    OnRightArrowClick();
                }
            }

            else
            {
                val = InputManager.Instance.GetControllerAxisUnique("Controller Left Stick X");

                if (val != 0.0f)
                {
                    if (val < 0.0f)
                    {
                        //Moved the stick left
                        OnLeftArrowClick();
                    }

                    else if (val > 0.0f)
                    {
                        //Moved the stick right
                        OnRightArrowClick();
                    }
                }

                else if (Input.GetKeyDown(KeyCode.Return) || Input.GetKeyDown("joystick button 0"))
                {
                    //Selected so play the sound and call the event
                    AudioManager.Instance.PlaySFX(AudioManager.Instance.UISelectClip);
                    onClick.Invoke();
                    return;
                }
            }
        }
    }

    private void OnInputStateChanged(object sender, InputManager.InputStateChangedEventArgs e)
    {
        if (e.NewState == Constants.InputState.MouseAndKeyboard || PageManager.CurrentPage.DefaultFocusedControl == null || PageManager.CurrentPage.DefaultFocusedControl.gameObject == null || PageManager.CurrentPage.DefaultFocusedControl.gameObject != gameObject)
        {
            //We're using mouse and keyboard, don't have a page or aren't the focused control so we've lost focus
            OnLostFocus();
        }

        else
        {
            if (e.NewState == Constants.InputState.Controller)
            {
                //We're using a controller and must have focus so call OnFocus
                OnFocus();
            }
        }
    }

    private void OnLeftArrowClick()
    {
        //Play the animation and audio, increment the indexes and update the displayed sprite
        _LeftArrowAnimator.SetTrigger("Pressed");
        if (CurrentIndex + 1 == Sprites.Length)
        {
            CurrentIndex = -1;
        }

        CurrentIndex++;
        SelectedSprite = Sprites[CurrentIndex];
        Image.sprite = SelectedSprite;
        AudioManager.Instance.PlaySFX(AudioManager.Instance.UIFocusClip);
    }

    private void OnRightArrowClick()
    {
        //Play the animation and audio, decrement the indexes and update the displayed sprite
        _RightArrowAnimator.SetTrigger("Pressed");
        if (CurrentIndex - 1 < 0)
        {
            CurrentIndex = Sprites.Length;
        }

        CurrentIndex--;
        SelectedSprite = Sprites[CurrentIndex];
        Image.sprite = SelectedSprite;
        AudioManager.Instance.PlaySFX(AudioManager.Instance.UIFocusClip);
    }

    private void OnFocus()
    {
        if (_Animator != null)
        {
            //Reset the triggers
            _Animator.ResetTrigger("OnLostFocus");
            _Animator.SetTrigger("OnFocus");
        }
    }

    private void OnLostFocus()
    {
        if (_Animator != null)
        {
            //Reset the triggers
            _Animator.ResetTrigger("OnFocus");
            _Animator.SetTrigger("OnLostFocus");
        }
    }

    public override void OnPointerEnter(PointerEventData data)
    {
        OnFocus();
    }

    public override void OnPointerExit(PointerEventData data)
    {
        OnLostFocus();
    }

    public override void OnSelect(BaseEventData data)
    {
        OnFocus();
        base.OnSelect(data);
    }

    public override void OnDeselect(BaseEventData data)
    {
        OnLostFocus();
        base.OnDeselect(data);
    }

    protected override void OnDestroy()
    {
        //Unsubscribe from events
        if (InputManager.Instance != null)
        {
            InputManager.Instance.InputStateChanged -= OnInputStateChanged;
        }

        base.OnDestroy();
    }
}
